#INCLUDE P16F628A.inc
; General and special stack based arithmetic functions.  
; Several functions are based on:
;
;*** SIGNED 32-BIT INTEGER MATHS ROUTINES FOR PIC16 SERIES BY PETER HEMSLEY ***
;
;Functions:
;	add
;	subtract
;	multiply
;	divide
;	round
;
; These were NOT implemented: sqrt, bin2dec, dec2bin
; 
; Added functions:
;	MidMul - multiply a mid point number by an integer giving an integer
;	MidDiv - divide an integer into an integer giving a mid point number
;	Shift - speed up multiply and divide by powers of 2 
;	Stack operations. Simplifies long calculations
;
; The original routines mostly used lower case for instructions. Additions or
; changes are mostly in upper case. Almost all the changes are to the division
; routine where the role of REGB and REGW are reversed so the reminder is
; left in REGB (if changing the register name was the only modification the
; instruction is still in lower case). The round routine was mostly rewritten
; to save duplicating existing code. Apart from divide, return is via
; PopStk. PopStk does not affect the state of the C flag.   
;
; IMPORTANT: these routines assume RP0/1 are set to Bank 1 by the caller
;
	global	    MpushLit	; push W onto the stack as +ve number
	global	    Mpush2U	; push 2 byte unsigned variable
	global	    Mpush2	; push 2 byte signed variable
	global	    Mpush3U	; push 3 byte unsigned variable
	global	    Mpush3	; push 3 byte signed variable
	global	    Mpush4	; push 4 byte signed variable
	global	    Mpop4	; pop 4 byte variable
	global	    Mpop3	; pop 3 byte variable
	global	    Madd
	global	    Msubtract
	global	    Multiply
	global	    MidMul	; special purpose multiply
	global	    MidDiv	; special purpose divide
	global	    Mdivide
	global	    Mshift	; generic shift of top of stack

;
; Note that for the push and pop routines, the limit conditions expect the
; stack to be at address 0xA0 and be 16 bytes long. If a different address
; or size is desired, the limit conditions need changing
;

StackAddr	EQU	0xA0	; the EQUs to define REGA/B/D would
StackSiz	EQU	4	; Stack Size in 32bit numbers (see above)

; Maths work area all in Bank 1

bnk1A		udata	StackAddr

STACK		RES	4*StackSiz
;
; The stack can be considered as an array of StackSiz columns of 4 rows.
; numbers are stored in columns - the bytes are StackSiz apart, not adjacent.
; The following EQUs are used to make the code more readable

; REGB is the 'top' of the stack, REGA is the 'first down'. Arithmetic
; is performed on REGA and REGB producing a result in REGA. Usually, each
; function pops the stack so the result overwrites REGB (top of stack).
; Shift operates on REGB and does not pop. Division leaves the remainder
; in REGB and the dividend in REGA. Both results can be popped or a call
; to Mround pops the remainder and rounds the dividend.
; [NOTE: for this program, the remainder is not required.
; Division always drops thru to the rounding function.]
; 
REGA0		EQU	StackAddr+1		; lsb
REGA1		EQU	StackAddr+(StackSiz + 1)
REGA2		EQU	StackAddr+(StackSiz * 2 + 1)
REGA3		EQU	StackAddr+(StackSiz * 3 + 1)

REGB0		EQU	StackAddr		; lsb
REGB1		EQU	StackAddr+StackSiz
REGB2		EQU	StackAddr+StackSiz * 2
REGB3		EQU	StackAddr+StackSiz * 3

; REGC is the destination BEFORE push, ends up in REGB after push
REGC0		EQU	StackAddr+(StackSiz * 4 - 1)
REGC1		EQU	StackAddr+(StackSiz - 1)
REGC2		EQU	StackAddr+(StackSiz * 2 - 1)
REGC3		EQU	StackAddr+(StackSiz * 3 - 1)

; REGW is a work register (not part of the stack) used in multiply
; and divide.
REGW0		RES	1	; lsb
REGW1		RES	1	; REGW 
REGW2		RES	1
REGW3		RES	1

MTEMP		RES	1	; work area, usually keeps the sign
MCOUNT		RES	1	;  "    "  , usually a loop counter
;=======================================================================
      	    code
	    
	    errorlevel -302 ; Inhibit banking error messages

;##################### push routines ##################
;
; Push entry points for various size variables.
; Pushx (x=1 to 4) expect signed variables
; PushxU (x=2 to 3) expect unsigned variables
;
MpushLit:
	MOVWF	MTEMP	; entry to push 1 byte unsigned constants
	MOVLW	MTEMP	; values over 127 would be interpreted
	MOVWF	FSR	; as -ve
	MOVLW	2   ; 1*2+0 1 byte unsigned
	GOTO	genpush

Mpush2U:
	MOVWF   FSR
	MOVLW	4 ; 2*2+0 2 byte unsigned
	GOTO	genpush

Mpush2:
	MOVWF   FSR
	MOVLW	5 ; 2*2+1 2 byte signed
	GOTO	genpush

Mpush3U:
	MOVWF   FSR
	MOVLW	6 ; 2*3+0 3 byte unsigned
	GOTO	genpush

Mpush3:
	MOVWF   FSR
	MOVLW	7 ; 2*3+1 3 byte signed
	GOTO	genpush

Mpush4:
	MOVWF   FSR
	MOVLW	9 ; 2*4+1 4 byte always signed

genpush:
; Transfers a new value to the stack with sign extension if required.
; The stack is barrel rolled 1 byte so the new value appears at REGB.
; The location of the value to push is held in FSR, the length specified
; by the least significant 4 bits of W. The remaining bits of W are
; unused but may have a future use so they are ANDed off.
;
; first, copy the new value to the stack where the data is about
; to be 'pushed out'. It will be pushed to the location of REGB.
	MOVWF	MCOUNT
	BCF     STATUS,C
	RRF     MCOUNT,W    ; strip off the signed or unsigned
	DECF	FSR,F		; move the pointer from the first byte
	ADDWF	FSR,F		; to the msb (the byte with the sign)
	BCF     STATUS,C
	RRF     MCOUNT,F    ; strip signed or unsigned and reveal the count
	BTFSC   STATUS,C    ; C clear, no sign test 
	RLF 	INDF,W      ; get msb
	CLRW                ; assume +ve
	BTFSC	STATUS,C	; skip if +ve (W=0)
	MOVLW	0xFF
	MOVWF	REGC0		; fill the destination with sign bits
	MOVWF	REGC1		; except the MSB which will be always be
	MOVWF	REGC2		; overwritten by the following shifts
MUnext:
	MOVF	REGC2,W  	; shift destination register up one byte
	MOVWF	REGC3
	MOVF	REGC1,W
	MOVWF	REGC2
	MOVF	REGC0,W
	MOVWF	REGC1
	MOVF	INDF,W  	; get a byte of the incoming number
	MOVWF	REGC0 	; into LSB
	DECF	FSR,F		; repeat until all bytes moved
	DECFSZ  MCOUNT,F
	GOTO	MUnext
; roll the stack
	MOVLW	STACK+1
	MOVWF	FSR
	MOVF	STACK,W	; get the first byte of stack
MPu2:
	XORWF	INDF,W 	; 3 XORs swaps W and F
	XORWF	INDF,F	; Each byte in STACK is moved up 1 byte
	XORWF	INDF,W	; in memory, effectively moving each 4 byte
	INCF	FSR,F	; value to the next column
	BTFSS	FSR,4	; FSR ok from A1 to AF, leave when B0
	GOTO	MPu2
	MOVWF	STACK	; save what was the last byte of stack in first byte
; all done
	RETURN

;##################### pop routines ##################
;
; Pop entry points for 3 or 4 byte variables.
; When only 3 bytes are popped, no test is made to see if
; the truncated byte was significant (not FF or 00)
;
Mpop4:
	MOVWF   FSR
	MOVLW	4	; 4 byte value returned
	GOTO	Mpop
Mpop3:
	MOVWF   FSR
	MOVLW	3	; 3 byte value returned

Mpop:
; transfers the value in REGB (top of stack) to a destination.
; The destination of the value to pop is held in FSR,
; the length specified by the least significant 4 bits of W.
; if length is less than 4, the most significant byte(s) is
; truncated. The stack is then popped (moved down 1 byte).
	ANDLW	0x0F		; bytes to move
	MOVWF	MCOUNT
MOnext:
	MOVF	STACK,W	; the next byte to move
	MOVWF	INDF	; into a destination byte
	MOVF	REGB1,W	; Shift REGB down one byte
	MOVWF	REGB0
	MOVF	REGB2,W
	MOVWF	REGB1
	MOVF	REGB3,W
	MOVWF	REGB2
	INCF	FSR,F	; pointer to next destination
	DECFSZ  MCOUNT,F; all moved?
	GOTO	MOnext
; value moved - pop the stack down
PopStk:
; this entry point also used by the arithmetic functions to
; overwrite REGB (top of stack) with the result in REGA
	MOVLW	STACK+(StackSiz*4-2)
	MOVWF	FSR
	MOVF	STACK+(StackSiz*4-1),W ; last byte of stack
MPd2:
	XORWF	INDF,W	; The value of of each byte is moved
	XORWF	INDF,F	; down 1 byte in memory, effectively
	XORWF	INDF,W	; moving each value to the previous
	DECF	FSR,F	; column
	BTFSS	FSR,4	; FSR ok from AE to A0, leave when 9F
	GOTO	MPd2
; all done
	RETURN
;
;##################### arithmetic routines ##################
;
;*** 32 BIT SIGNED SUBTRACT ***
;REGA - REGB -> REGA
;Return carry set if overflow

Msubtract
	call	negateb		;Negate REGB
	skpnc
	GOTO	PopStk		;Overflow

;*** 32 BIT SIGNED ADD ***
;REGA + REGB -> REGA
;Return carry set if overflow

Madd
	movf	REGA3,w		;Compare signs
	xorwf	REGB3,w
	movwf	MTEMP

	call	addba		;Add REGB to REGA

	clrc			;Check signs
	movf	REGB3,w		;If signs are same
	xorwf	REGA3,w		;so must result sign
	btfss	MTEMP,7		;else overflow
	addlw	0x80
	GOTO	PopStk

;*** 32 BIT SIGNED MULTIPLY ***
;REGA * REGB -> REGA
;Return carry set if overflow

Multiply
	clrf	MTEMP		;Reset sign flag
	call	absa		;Make REGA positive
	skpc
	call	absb		;Make REGB positive
	skpnc
	GOTO	PopStk	;Overflow
	CALL    movclr ; move REGA to REGW and clear REGA
	movlw	D'31'		;Loop counter
	movwf	MCOUNT

muloop
	call	sla		;Shift left product and multiplicand
	rlf	REGW0,f	; code variation: this was in a subroutine,
	rlf	REGW1,f	; but was moved inline
	rlf	REGW2,f
	rlf	REGW3,f

	rlf	REGW3,w		;Test MSB of multiplicand
	skpnc			;If multiplicand bit is a 1 then
	call	addba		;add multiplier to product

	skpc			;Check for overflow
	rlf	REGA3,w
	skpnc
	GOTO	PopStk

	decfsz	MCOUNT,f	;Next
	goto	muloop

	btfsc	MTEMP,0		;Check result sign
	call	negatea		;Negative
	GOTO	PopStk

MidMul:
;
; Special purpose multiply of a signed mid point number -
; 1 integer byte 2 fractional bytes with a 3 byte integer to give an
; integer result (discarding the fractional part of the result).
; Used for converting detector units (the mid point number) to PWM units    
; using the calculated conversion value.
; The algorithm uses right shifts. As there are 24 shifts, the additions
; are to REGA1-3, REGA0 providing 8 bits of 'padding' so only 16 bits are
; 'lost' (being 16 bits of fraction but only the integer result required).
;
	CLRF	MTEMP		;Reset sign flag
	CALL	absa		;Make REGA positive
	BTFSS   STATUS,C
	CALL	absb		;Make REGB positive
	BTFSC   STATUS,C
	RETURN
; move REGA to REGW and clear REGA which will hold
; the integer part of the product
	CALL    movclr
	MOVLW	D'24'		;Loop counter
	MOVWF	MCOUNT
M24loop:
	RRF     REGB2,F ; get lo bit of multiplier
	RRF     REGB1,F
	RRF     REGB0,F
	BTFSS   STATUS,C
	GOTO    M24tst
	MOVF	REGW0,W		;Add lo byte
	ADDWF	REGA1,F
	MOVF    REGW1,W
	BTFSC   STATUS,C
	INCFSZ	REGW1,W		;Add carry_in
	ADDWF   REGA2,F
	MOVF    REGW2,W
	BTFSC   STATUS,C
	INCF 	REGW2,W		;Add carry_in
	ADDWF   REGA3,F
M24tst:
; either C is clear (from test of lo bit of multiplier)
; or C has carry out from the ADDWF to REGA3
	RRF     REGA3,F
	RRF     REGA2,F
	RRF     REGA1,F
	RRF     REGA0,F
	DECFSZ	MCOUNT,F	;Next
	GOTO    M24loop
; Discard the multiplier (moves the product to REGB)
	BCF     STATUS,C
	BTFSC	MTEMP,0		; Check if result needs sign correction
	CALL	negatea		; Negate if needed
        GOTO	PopStk


; *** Special purpose division ***
MidDiv:
; divide samples into a variable returning a mid point number
; 16 bit integer and 16 bit fraction. This is used:
; 1 - to calculate a change per second from a change per period
; 2 - to calculate average error per second from accumulated
;     error for the period
	movlw   D'47'
	goto    divmain

;*** 32 BIT SIGNED DIVIDE ***
;REGA / REGB -> REGA
;Remainder in REGB
;Return carry set if overflow or division by zero

Mdivide
	movlw	D'31'		;Loop counter
divmain:
	movwf	MCOUNT
	clrf	MTEMP		;Reset sign flag
	call	absb		;Make divisor (REGB) positive
	skpnc
	RETURN
; modification - so the remainder ends up on the stack, REGB is moved to
; REGW. The use of REGB and REGW is the opposite of the original code but
; the logic remains the same
	MOVF	REGB0,w		; Move REGB (divisor) to REGW at the
	MOVWF	REGW0       ; same time test for zero divisor
	MOVF	REGB1,w
	MOVWF	REGW1
	MOVF	REGB2,w
	MOVWF	REGW2
	MOVF	REGB3,w
	MOVWF	REGW3
	IORWF	REGW2,w
	IORWF	REGW1,w
	IORWF	REGW0,w
;
	sublw	0   ; if all zero, will set C
	skpc
	call	absa		;Make dividend (REGA) positive
	skpnc
	RETURN
; clear REGB to take the remainder
	clrf	REGB0		;Clear remainder
	clrf	REGB1
	clrf	REGB2
	clrf	REGB3
	call	sla		;Purge sign bit
dvloop
	call	sla		;Shift dividend (REGA) msb into remainder (REGB)
	CALL	SlbTst	; shifts and tests remainder > divisor
	skpc			;Carry set if remainder >= divisor
	goto	dremlt
	movf	REGW0,w		;Subtract divisor (REGW) from remainder (REGB)
	subwf	REGB0,f
	movf	REGW1,w
	skpc
	incfsz	REGW1,w
	subwf	REGB1,f
	movf	REGW2,w
	skpc
	incfsz	REGW2,w
	subwf	REGB2,f
	movf	REGW3,w
	skpc
	incfsz	REGW3,w
	subwf	REGB3,f
	clrc
	bsf	REGA0,0		;Set quotient bit
dremlt
	decfsz	MCOUNT,f	;Next
	goto	dvloop
	btfsc	MTEMP,0		;Check result sign
	call	negatea		;Negative
;	RETURN    ; We don't want reminder, drop thru to round

;*** ROUND RESULT OF DIVISION TO NEAREST INTEGER ***

; Mround:
; modified from original. Some code duplication was noticed so some was
; put in subroutine SlbTst and the IncA entry to negatea was added.
; No error testing, should not be capable of creating an error
	clrf	MTEMP		;Reset sign flag
	call	absa		;Make positive
	clrc
	CALL	SlbTst      ; shifts and tests remainder > divisor
	CLRW			; prevent IncA from returning an error
	BTFSC	STATUS,C		;Carry set if remainder >= divisor
	CALL	IncA ; Increment REGA
	btfsc	MTEMP,0		;Restore sign
	call	negatea
	GOTO	PopStk

Mshift:
; shift REGB left or right, count in W (transferred to FSR by SaveStat)
; If W is +ve, left shift. If -ve, right shift
	IORLW   0x00 ; test for zero
	BTFSC   STATUS,Z ; check if no adjustment needed
	RETURN
	MOVWF   MCOUNT
	BTFSS   MCOUNT,7
	GOTO    NormL
; -ve - right shift (divide by 2)
NormR:
	RLF	REGB3,W ; put the sign bit in C
	RRF	REGB3,F
	RRF	REGB2,F
	RRF	REGB1,F
	RRF	REGB0,F
	INCFSZ  MCOUNT,F
	GOTO    NormR
;	GOTO    LoadStat
	RETURN
NormL:
; +ve - left shift (multiply by 2)
	BCF	STATUS,C
	RLF	REGB0,F
	RLF	REGB1,F
	RLF	REGB2,F
	RLF	REGB3,F
	DECFSZ  MCOUNT,F
	GOTO    NormL
	RETURN

;UTILITY ROUTINES

;Add REGB to REGA (Unsigned)
;Used by add, multiply,

addba	movf	REGB0,w		;Add lo byte
	addwf	REGA0,f
	movf	REGB1,w		;Add mid-lo byte
	skpnc			;No carry_in, so just add
	incfsz	REGB1,w		;Add carry_in to REGB
	addwf	REGA1,f		;Add and propagate carry_out
	movf	REGB2,w		;Add mid-hi byte
	skpnc
	incfsz	REGB2,w
	addwf	REGA2,f
	movf	REGB3,w		;Add hi byte
	skpnc
	incfsz	REGB3,w
	addwf	REGA3,f
	return

;Check sign of REGA and convert negative to positive
;Used by multiply, divide, round

absa	rlf	REGA3,w
	skpc
	return			;Positive
;Negate REGA
;Used by absa, multiply, divide, round
negatea	movf	REGA3,w		;Save sign in w
	andlw	0x80

	comf	REGA0,f		;2's complement
	comf	REGA1,f
	comf	REGA2,f
	comf	REGA3,f
	incf	MTEMP,f		;flip sign flag
IncA ; new entry point from round routine
	incfsz	REGA0,f
	goto	nega1
	incfsz	REGA1,f
	goto	nega1
	incfsz	REGA2,f
	goto	nega1
	incf	REGA3,f
nega1
	addwf	REGA3,w		;Return carry set if -2147483648
	return

;Check sign of REGB and convert negative to positive
;Used by multiply, divide

absb	rlf	REGB3,w
	skpc
	return			;Positive

;Negate REGB
;Used by absb, subtract, multiply, divide

negateb	movf	REGB3,w		;Save sign in w
	andlw	0x80

	comf	REGB0,f		;2's complement
	comf	REGB1,f
	comf	REGB2,f
	comf	REGB3,f
	incfsz	REGB0,f
	goto	negb1
	incfsz	REGB1,f
	goto	negb1
	incfsz	REGB2,f
	goto	negb1
	incf	REGB3,f
negb1
	incf	MTEMP,f		;flip sign flag
	addwf	REGB3,w		;Return carry set if -2147483648
	return

;Move REGA to REGW
;Clear REGA
;Used by multiply

movclr:
	movf	REGA0,w		;move multiplier to REGW
	movwf	REGW0
	movf	REGA1,w
	movwf	REGW1
	movf	REGA2,w
	movwf	REGW2
	movf	REGA3,w
	movwf	REGW3

	clrf	REGA0		;Clear REGA to receive product
	clrf	REGA1
	clrf	REGA2
	clrf	REGA3
	RETURN

SlbTst:
;
; code modification: moved from divide, used by divide, round
; shifts remainder - when dividing, shifts in a bit from REGA;
; if rounding, a zero bit . Then tests remainder => divisor
;
	rlf	REGB0,f ; shift
	rlf	REGB1,f
	rlf	REGB2,f
	rlf	REGB3,f
	movf	REGW3,w	; Test
	subwf	REGB3,w
	skpz
	RETURN
	movf	REGW2,w
	subwf	REGB2,w
	skpz
	RETURN
	movf	REGW1,w
	subwf	REGB1,w
	skpz
	RETURN
	movf	REGW0,w
	subwf	REGB0,w
	RETURN

;Shift left REGA
;Used by multiply, divide, round

sla	rlf	REGA0,f
	rlf	REGA1,f
	rlf	REGA2,f
	rlf	REGA3,f
	return
	
	end